/*
 * Copyright (c) 1999-2001 Lutris Technologies, Inc. All Rights
 * Reserved.
 * 
 * This source code file is distributed by Lutris Technologies, Inc. for
 * use only by licensed users of product(s) that include this source
 * file. Use of this source file or the software that uses it is covered
 * by the terms and conditions of the Lutris Enhydra Development License
 * Agreement included with this product.
 * 
 * This Software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
 * ANY KIND, either express or implied. See the License for the specific terms
 * governing rights and limitations under the License.
 * 
 * Contributor(s):
 * 
 * $Id: DefaultParser.java,v 1.1.1.1 2001/01/05 05:11:17 sese0235 Exp $
 */

package de.kxml.parser;

import java.io.*;
import java.util.*;
import de.kxml.*;
import de.kxml.io.*;

/**
 * A simple, pull based Common XML parser
 */
public class DefaultParser extends Parser {
    ParseEvent      next;
    LookAheadReader reader;

    // Hashtable prefixMap = new Hashtable ();
    // Vector openTags = new Vector ();
    Vector	    qNames = new Vector();
    boolean	    immediateClose = false;
    StartTag	    current;

    /**
     * creates a new Parser based on the give reader
     */
    public DefaultParser(Reader reader) throws IOException {
	this.reader = new LookAheadReader(reader);

	read();
    }

    /**
     * amp is already consumed, consume incl. comma
     */
    char parseCharacterEntity() throws IOException {
	String code = reader.readTo(';');

	reader.read();

	if (code.charAt(0) == '#') {
	    return code.charAt(1) == 'x' 
		   ? (char) Integer.parseInt(code.substring(2), 16) 
		   : (char) Integer.parseInt(code.substring(1));
	} else if (code.equals("lt")) {
	    return '<';
	} else if (code.equals("gt")) {
	    return '>';
	} else if (code.equals("apos")) {
	    return '\'';
	} else if (code.equals("quot")) {
	    return '"';
	} else if (code.equals("amp")) {
	    return '&';
	} else {
	    throw new ParseException("Illegal character entity: &" + code 
				     + ";", reader);
	}
    } 

    /* precondition: &lt;!- consumed */

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    LegacyEvent parseComment() throws IOException {
	StringBuffer buf = new StringBuffer();

	if (reader.read() != '-') {
	    throw new ParseException("'-' expected", reader);
	} 

	int cnt;
	int lst;

	while (true) {
	    buf.append(reader.readTo('-'));

	    if (reader.read() == -1) {
		throw new ParseException("Unexpected end of file while reading commnet", 
					 reader);
	    } 

	    cnt = 0;

	    do {
		lst = reader.read();
		cnt++;    // adds one more, but first is not cnted
	    } while (lst == '-');

	    if (lst == '>' && cnt >= 2) {
		break;
	    } 

	    while (cnt-- > 0) {
		buf.append('-');
	    }

	    buf.append((char) lst);
	} 

	while (cnt-- > 2) {
	    buf.append('-');
	}

	return new LegacyEvent(Xml.COMMENT, buf.toString());
    } 

    /* precondition: &lt! consumed */

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    LegacyEvent parseDoctype() throws IOException {
	StringBuffer buf = new StringBuffer();
	int	     nesting = 1;

	while (true) {
	    int i = reader.read();

	    switch (i) {

	    case -1:
		throw new ParseException("unexpected eof", reader);

	    case '<':
		nesting++;

		break;

	    case '>':
		if ((--nesting) == 0) {
		    return new LegacyEvent(Xml.DOCTYPE, buf.toString());
		} 

		break;
	    }

	    buf.append((char) i);
	} 
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    TextEvent parseCData() throws IOException {
	if (!reader.readTo('[').equals("CDATA")) {
	    throw new ParseException("Invalid CDATA start!", reader);
	} 

	reader.read();    // skip '['

	StringBuffer buf = new StringBuffer();
	int	     c0 = reader.read();
	int	     c1 = reader.read();

	while (true) {
	    int c2 = reader.read();

	    if (c2 == -1) {
		throw new ParseException("unexpected eof", reader);
	    } 

	    if (c0 == ']' && c1 == ']' && c2 == '>') {
		break;
	    } 

	    buf.append(c0);

	    c0 = c1;
	    c1 = c2;
	} 

	return new TextEvent(buf.toString());
    } 

    /* precondition: &lt;/ consumed */

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    EndTag parseEndTag() throws IOException {
	String name = reader.readTo('>');

	reader.read();    // skip >

	int    last = qNames.size() - 1;
	String qName = (String) qNames.elementAt(last);

	qNames.removeElementAt(last);

	if (!qName.equals(name)) {
	    throw new ParseException("StartTag <" + qName 
				     + "> does not match end tag </" + name 
				     + ">", reader);
	} 

	EndTag result = new EndTag(current);

	current = current.previous;

	return result;
    } 

    /**
     * precondition: <? consumed
     */
    ParseEvent parsePI() throws IOException {
	StringBuffer buf = new StringBuffer(reader.readTo('?'));

	reader.read();    // ?

	while (reader.peek() != '>') {
	    buf.append('?');

	    int r = reader.read();

	    if (r == -1) {
		throw new ParseException("Unexpected eof while reading PI", 
					 reader);
	    } 

	    buf.append((char) r);
	    buf.append(reader.readTo('?'));
	    reader.read();
	} 

	reader.read();    // consume >

	return new LegacyEvent(Xml.PROCESSING_INSTRUCTION, buf.toString());
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    StartTag parseStartTag() throws IOException {
	current = new StartTag(current, reader);

	String qName = current.name;

	if (processNamespaces) {
	    try {
		current.fixNamespaces();
	    } catch (Exception e) {
		throw new ParseException(e, reader);
	    } 
	} 

	// System.out.println ("tag: ("+next+")");
	immediateClose = current.degenerated;

	if (!immediateClose) {
	    qNames.addElement(qName);
	} 

	return current;
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    TextEvent parseText() throws IOException {
	StringBuffer buf = new StringBuffer();
	int	     nextChar;

	do {
	    nextChar = reader.read();

	    if (nextChar == '&') {
		buf.append(parseCharacterEntity());
	    } else {
		buf.append((char) nextChar);
	    }

	    nextChar = reader.peek();
	} while (nextChar != '<' && nextChar != -1);

	return new TextEvent(buf.toString());
    } 

    /**
     * precondition: &lt; consumed
     */
    void parseSpecial() throws IOException {
	switch (reader.peek()) {

	case -1:
	    throw new ParseException("Unexpected end of file after reading <", 
				     reader);

	case '!':
	    reader.read();

	    switch (reader.peek()) {

	    case '-':
		reader.read();

		next = parseComment();

		break;

	    case '[':
		reader.read();

		next = parseCData();

		break;

	    default:
		next = parseDoctype();

		break;
	    }

	    break;

	case '?':
	    reader.read();

	    next = parsePI();

	    break;

	case '/':
	    reader.read();

	    next = parseEndTag();

	    break;

	default:
	    next = parseStartTag();
	}
    } 

    /**
     * returns the next parse event without consuming it
     */
    public ParseEvent peek() {
	return next;
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    public ParseEvent read() throws IOException {
	ParseEvent result = next;

	if (immediateClose) {
	    next = new EndTag(current);
	    current = current.getPrevious();
	    immediateClose = false;
	} else {
	    switch (reader.peek()) {

	    case '<':
		reader.read();
		parseSpecial();

		break;

	    case -1:
		if (current != null) {
		    throw new ParseException("End tag missing for: " 
					     + current, reader);
		} 

		next = new EndDocument();

		// System.out.println ("END Reached!");
		break;

	    default:
		next = parseText();
	    }
	} 

	return result;
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @see
     */
    public int getLineNumber() {
	return reader.getLineNumber();
    } 

}

